# qoa-format

A JavaScript port of [The "Quite OK Audio" (QOA) format](https://github.com/phoboslab/qoa), a lossy audio compression that achieves relatively decent compression with fast decoding and not much complexity. Also see [this](https://phoboslab.org/log/2023/02/qoa-time-domain-audio-compression) blog post by @phoboslab.

Features:

- Lossy, time domain audio compression format with a constrained bitrate of 277 kbits/s for stereo 44100 Hz
- This pure JavaScript encoder / decoder implementation bundles to about 6kb minified (or 3kb for just the decoder)
- Works in Node.js and the browser without any specific audio APIs
- Rough decoding performance tests on MBP M1 Max in Chrome show similar performance to WebAudio `decodeAudioData`
- The audio format and encoder/decoder is open source and MIT licensed

_This software is still experimental, unstable, and likely to change or break. Use at your own risk._

## ⚠️ Warning

If you are hacking and experimenting with this, take care when using headphones as garbage data may produced loud noises that can damage hearing.

As the spec is still in flux, this may not correctly decode files generated by different versions of [the C encoder](https://github.com/phoboslab/qoa/) or third-party implementations. This has been tested against [this particular QOA commit tree](https://github.com/phoboslab/qoa/tree/e8386f41d435a864ce2890e9f56d964215b40301) when building from source on macOS.

## Example

```js
import { encode, decode } from 'qoa-format';

// or just the decode/encode function -
// import decode from 'qoa-format/decode.js';

// Encoding raw audio samples
const audioData = {
  sampleRate: 44100
  channelData: [
    new Float32Array([ /* audio samples */ ])
  ]
}

// lossy encode raw audio samples to Uint8Array QOA file
const qoaFile = encode(audioData)

// Decode QOA file back to audio data
const decodedAudio = decode(qoaFile);

// Show info about the decoded file
console.log(decodedAudio.sampleRate); // 44100
console.log(decodedAudio.channels); // 1
console.log(decodedAudio.samples); // (number of samples in audio signal)
console.log(decodedAudio.channelData); // [ Float32Array(samples) ]
```

See [test/sine.js](./test/sine.js) for encoding/decoding a 441 Hz sine wave, and [test/webaudio.js](./test/webaudio.js) for a decoder and web audio QOA player.

## Install

Use [npm](https://npmjs.com/) to install.

```sh
npm install qoa-format --save
```

## API Usage

#### `data = encode({ channelData, sampleRate })`

Encodes the audio signal in `channelData` with `sampleRate` as a QOA file, returning `data` as a `Uint8Array`. The length of `channelData` determines the number of channels (mono, stereo, multi-channel), and each element is expected to be a `Float32Array` with the same length (i.e. `samples` or number of sample frames). The signal is expected to range from `-1..1`.

#### `audio = decode(Uint8Array | Buffer)`

Decodes the `Uint8Array` or `Buffer` object into an `audio` specifier which has the following format:

```
{
  sampleRate, // in Hz
  channels, // number of channels
  samples, // number of frame samples per channel
  channelData: [
    // an array of audio samples for each channel
    Float32Array(samples),
    ...
  ]
}
```

## Run from Source

Once cloned, you can `npm install` and then run the test:

```sh
npm run test
```

Or the HTML/WebAudio test, run the below command and then open [http://localhost:8000/test/webaudio.html](http://localhost:8000/test/webaudio.html). **Note:** Take caution when wearing headphones if you are hacking with this. 🎧

```sh
npm run test:browser
```

Or run the CLI to convert files:

```sh
# encode WAV to QOA
node test/cli.js input.wav output.qoa

# decode QOA to WAV
node test/cli.js output.qoa converted.wav
```

## TODOs

- A more optimized BitStream reader/writer could be added to this module to remove a dependency, produce a smaller build size, and potentially improve performance.
- The QOA format is easily seekable, this should be exposed in the API to allow fast seeking and audio streaming on the web

## Credits

- QOA format by @phoboslab
- JavaScript port by @mattdesl
- Thanks to @thi.ng for the BitStream decoder

## License

MIT, see [LICENSE.md](http://github.com/mattdesl/qoa-format/blob/master/LICENSE.md) for details.
